/*

$Id: bgp-update-create.c,v 1.3 2003/04/28 23:06:30 peloy Exp $

Creates the TCP payload corresponding to a BGP update message. RFC 1771
is your friend.

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <getopt.h>
#include <netdb.h>

#include "bgp.h"
#include "bgp_attr.h"
#include "bgp_aspath.h"

/*
 * Need these tricks because we'll get bus errors on some
 * architectures (Sparc, 68k) if we just write/read 16 or 32-bit
 * values to non word-aligned addresses.
 */
#define U_INT16_WRITE(ptr, value) \
{ \
	u_int8_t tmp[2]; \
	*(u_int16_t *) tmp = (value); \
	( (u_int8_t *) (ptr) )[0] = tmp[0]; \
	( (u_int8_t *) (ptr) )[1]= tmp[1]; \
}

#define U_INT32_WRITE(ptr, value) \
{ \
	u_int8_t tmp[4]; \
	*(u_int32_t *) tmp = (value); \
	( (u_int8_t *) (ptr) )[0] = tmp[0]; \
	( (u_int8_t *) (ptr) )[1] = tmp[1]; \
	( (u_int8_t *) (ptr) )[2] = tmp[2]; \
	( (u_int8_t *) (ptr) )[3] = tmp[3]; \
}

#define U_INT16_READ(ptr) ( ( ( ( (u_int8_t *) (ptr) )[0]) << 8) + \
						    ( ( ( (u_int8_t *) (ptr) )[1]) & 0xff) )

void pdie(const char *);
void dump(const unsigned char *, unsigned);

struct bgphdr {
	u_int8_t marker[16];
	u_int16_t length;
	u_int8_t type;
};

struct options {
	u_int16_t as;
	u_int32_t next_hop;
	u_int8_t prefix;
	u_int32_t destination_net;
} options = {
	0,
	0,
	0,
	0
};

char *program_name;

#define USAGE \
"Usage: %s [--help] --as <n> --nexthop <hostname>\n" \
"       --destnet <net/prefix>\n\n" \
"--as:      Autonomous System number\n" \
"--nexthop: Host name or IP address of next hop\n" \
"--destnet: Network number/prefix to advertise\n"

void usage(void)
{
	fprintf(stderr, USAGE, program_name);
	exit(1);
}

int main(int argc, char **argv)
{
	int c;

	struct bgphdr *bgp;
	u_int8_t *bgp_payload;

	u_int16_t *unfeasible_routes_len_ptr;
	u_int16_t *tot_path_attr_len_ptr;
	u_int8_t *path_attr_ptr;
	u_int8_t *net_reachability_ptr;

	int i;
	struct hostent *host;

	if ((program_name = strrchr(argv[0], '/')) )
		program_name++;
	else
		program_name = argv[0];

	while (1) {
		int option_index = 0;
		static struct option long_options[] = {
			{"as", 1, NULL, 'a'},
			{"nexthop", 1, NULL, 'h'},
			{"destnet", 1, NULL, 'n'},
			{"help", 0, NULL, '?'},
			{0, 0, NULL, 0}
		};
		c = getopt_long(argc, argv, "a:h:n:?", long_options,
				&option_index);

		if (c == -1)
			break;

		switch (c) {
			char *tmp;

			case 'a':
				options.as = atoi(optarg);
				break;

			case 'h':
				if ( (host = gethostbyname(optarg) ) == NULL) {
					fprintf(stderr, "Error resolving hostname.\n");
					exit(1);
				}

				options.next_hop = *(unsigned *)
					host->h_addr_list[0];
				break;

			case 'n':
				if ( (tmp = strchr(optarg, '/') ) == NULL) {
					fprintf(stderr, "What's the prefix?");
					exit(1);
				}
				*tmp++ = '\0';

				options.prefix = atoi(tmp);

				if (inet_aton(optarg,
					      (struct in_addr *)
					      &options.destination_net) == 0) {
					fprintf(stderr, "Invalid network address");
					exit(1);
				}
				options.destination_net = htonl(options.destination_net);
				break;

			default:
				usage();
		}
	}

	/* Make sure we have all the options we need */
	if (options.prefix == 0 || options.next_hop == 0 ||
	    options.as == 0 || options.destination_net == 0)
		usage();

	if ( (bgp = malloc(BGP_MAX_PACKET_SIZE) ) == NULL)
		pdie("Error allocating memory");

	bzero(bgp, BGP_MAX_PACKET_SIZE);

	/**************** BGP header ****************/
	memset(bgp->marker, 0xff, sizeof(bgp->marker) );
	bgp->type = BGP_MSG_UPDATE;

	/*
	 * Need to figure out why sizeof(struct bgphdr) is 20 instead
	 * of 19 -> must use 19 instead of sizeof(struct bgphdr).
	 */
	bgp_payload = (u_int8_t *) bgp + 19;

	/**************** Unfeasible routes ****************/
	unfeasible_routes_len_ptr = (u_int16_t *) bgp_payload;
	U_INT16_WRITE(unfeasible_routes_len_ptr, htons(0) );

	/**************** Path attributes ****************/
	tot_path_attr_len_ptr = (u_int16_t *) (bgp_payload +
				ntohs(U_INT16_READ(unfeasible_routes_len_ptr) ) +
				sizeof(u_int16_t) );

	path_attr_ptr = (u_int8_t *) tot_path_attr_len_ptr + sizeof(u_int16_t);

	/* Attribute #1: ORIGIN */
	*path_attr_ptr++ = BGP_ATTR_FLAG_TRANS; /* Attribute flags */
	*path_attr_ptr++ = BGP_ATTR_ORIGIN; /* Attribute type */
	*path_attr_ptr++ = 1; /* Length of this attribute */

	*path_attr_ptr++ = BGP_ORIGIN_IGP; /* Origin */

	/* Attribute #2: AS_PATH */
	*path_attr_ptr++ = BGP_ATTR_FLAG_TRANS; /* Attribute flags */
	*path_attr_ptr++ = BGP_ATTR_AS_PATH; /* Attribute type */
	*path_attr_ptr++ = 4; /* Length of this attribute */

	*path_attr_ptr++ = AS_SEQUENCE; /* Path segment type */
	*path_attr_ptr++ = 1; /* Path segment length = 1 AS */
	U_INT16_WRITE(path_attr_ptr, htons(options.as) ); /* Path segment value */
	path_attr_ptr+= sizeof(u_int16_t);

	/* Attribute #3: NEXT_HOP */
	*path_attr_ptr++ = BGP_ATTR_FLAG_TRANS; /* Attribute flags */
	*path_attr_ptr++ = BGP_ATTR_NEXT_HOP; /* Attribute type */
	*path_attr_ptr++ = 4; /* Length of this attribute */

	U_INT32_WRITE(path_attr_ptr, options.next_hop);
	path_attr_ptr += sizeof(u_int32_t);

	/* Attribute #4: MULTI_EXIT_DISC */
	*path_attr_ptr++ = BGP_ATTR_FLAG_OPTIONAL; /* Attribute flags */
	*path_attr_ptr++ = BGP_ATTR_MULTI_EXIT_DISC; /* Attribute type */
	*path_attr_ptr++ = 4; /* Length of this attribute */

	U_INT32_WRITE(path_attr_ptr, htonl(0) );
	path_attr_ptr += sizeof(u_int32_t);

	/* Finally, compute the length of all the attributes */
	U_INT16_WRITE(tot_path_attr_len_ptr, htons( (void *) path_attr_ptr -
				 (void *) tot_path_attr_len_ptr -
				 sizeof(u_int16_t) ) );

	/**************** Network layer reachability info.  ****************/

	net_reachability_ptr = path_attr_ptr;

	*net_reachability_ptr++ = options.prefix; /* Length of the IP address prefix */
	{
		u_int8_t tmp[4];

		*(u_int32_t *) tmp = htonl(options.destination_net);

		for (i = 0; i < options.prefix/8; i++)
			*net_reachability_ptr++ = tmp[i];
	}

	/* Finally, compute the length of the BGP message */
	bgp->length = htons( (void *) net_reachability_ptr -
		      (void *) bgp);

	for (i = 0; i < ntohs(bgp->length); i++)
		printf("%c", ( (u_int8_t *) bgp)[i]);

	free(bgp);

	return 0;
}

/**********************************************************************
 * pdie --- Call perror() to figure out what's going on and die.
 **********************************************************************/
void pdie(const char *mesg)
{
	perror(mesg);
	exit(1);
}

void dump(const unsigned char *data, unsigned len)
{
	unsigned i, j;

	for (i = 0; i <= len/16; i++) {
		printf("%08x  ", i*16);

		for (j = 0; j < 16; j++) {
			if (i*16 + j < len)
				printf("%02x", data[i*16 + j]);
			else
				printf("  ");

			if (j & 1)
				printf(" ");
		}

		for (j = 0; j < 16; j++)
			if (i*16 + j < len)
				printf("%c", isprint(data[i*16 + j]) ?
				       data[i*16 + j] : '.');

		printf("\n");
	}

	printf("\n");
}
